#!/bin/env python3

"""
Generate 100 framework subdirectories with some sample code in them.
"""

import subprocess
from pathlib import Path


def main():
    for folder_index in range(1, 101):
        folder_name = f'FW{folder_index}'
        folder_path = Path(folder_name)
        if not folder_path.exists():
            subprocess.run(['mkdir', folder_name])  # Ignore existing folders
        create_files(folder_name, folder_path)


def create_files(folder_name: str, folder_path: Path):
    # Umbrella header
    umbrella_path = folder_path / f'{folder_name}.h'
    with open(umbrella_path, 'w') as f:
        f.write(f"""
#import <Foundation/Foundation.h>
FOUNDATION_EXPORT double {folder_name}VersionNumber;
FOUNDATION_EXPORT const unsigned char {folder_name}VersionString[];
#import <{folder_name}/{folder_name}Functions.h>
#import <{folder_name}/{folder_name}Classes.h>
""")

    # Functions header
    functions_header_path = folder_path / f'{folder_name}Functions.h'
    with open(functions_header_path, 'w') as f:
        for function_number in range(1, 101):
            f.write(f'extern int {folder_name}Function{function_number}(int param);\n')

    # Functions implementation
    functions_implementation_path = folder_path / f'{folder_name}Functions.c'
    with open(functions_implementation_path, 'w') as f:
        for function_number in range(1, 101):
            f.write(f'int {folder_name}Function{function_number}(int param) {{ return param + 1; }}\n')

    # Classes header
    classes_header_path = folder_path / f'{folder_name}Classes.h'
    with open(classes_header_path, 'w') as f:
        f.write('''#import <Foundation/Foundation.h>

#define HUMN_IMPLEMENTED_IN_SWIFT(_module) \\
__attribute__((external_source_symbol(language="Swift", defined_in=#_module, generated_declaration)))

#define HUMN_CLASS_IMPLEMENTED_IN_SWIFT(_swift_name, _module) \\
__attribute__((swift_name(#_swift_name))) \\
__attribute__((objc_subclassing_restricted)) \\
HUMN_IMPLEMENTED_IN_SWIFT(_module)

''')
        for class_number in range(1, 51):
            class_name = f'{folder_name}Class{class_number}'
            f.write(f'''HUMN_CLASS_IMPLEMENTED_IN_SWIFT(Swift{class_name}, {folder_name})
@interface {folder_name}Class{class_number}: NSObject
''')
            for method_number in range(1, 11):
                f.write(f'- (int)class{class_number}Method{method_number}WithParam:(int)param;\n');
            f.write('@end\n\n')

    # Classes implementation
    classes_implementation_path = folder_path / f'{folder_name}Classes.swift'
    with open(classes_implementation_path, 'w') as f:
        f.write('import Foundation\n\n')

        for class_number in range(1, 51):
            class_name = f'{folder_name}Class{class_number}'
            f.write(f'''@objc({class_name})
public class Swift{class_name}: NSObject {{
''')
            for method_number in range(1, 11):
                f.write(f'''    @objc(class{class_number}Method{method_number}WithParam:)
    func class{class_number}method{method_number}(param: Int) -> Int {{ return param + 1; }}
''')
            f.write('}\n\n')


if __name__ == '__main__':
    main()
